const { askVertex } = require('../config/vertex');
const { sendMessageToUser } = require('./facebook');

/**
 * System prompt for the AI assistant
 */
const systemPrompt = `You are a helpful, polite assistant acting as the official agent of a Facebook Page.
- Answer briefly and clearly.
- Answer in the same language as the user (if user writes in Persian, answer in Persian).
- Do NOT say you are an AI, just act as the page's assistant.
- If you are not sure about pricing, delivery, or internal policies, ask a follow-up question or say you will check.`;

/**
 * Handles an incoming message from a Facebook user
 * Processes the message through Vertex AI and sends a response back
 * @param {string} senderId - The Facebook user ID (sender ID)
 * @param {string} userText - The message text from the user
 * @returns {Promise<void>}
 */
async function handleIncomingMessage(senderId, userText) {
  try {
    // Build the full prompt
    const fullPrompt = systemPrompt + '\n\nUser: ' + userText;

    // Get response from Vertex AI
    let reply = await askVertex(fullPrompt);

    // Fallback if reply is empty or too short
    if (!reply || reply.trim().length < 3) {
      reply = 'متاسفم، الان نتونستم خوب پاسخ بدم. لطفاً دوباره سؤال‌تان را بپرسید.';
    }

    // Send the reply back to the user
    await sendMessageToUser(senderId, reply);
  } catch (error) {
    console.error('Error in handleIncomingMessage:', error.message);
    console.error('Full error:', error);

    // Send fallback error message to user
    try {
      await sendMessageToUser(
        senderId,
        'متاسفم، یک خطای فنی رخ داد. لطفاً چند دقیقه بعد دوباره پیام بدهید.'
      );
    } catch (sendError) {
      console.error('Failed to send error message to user:', sendError.message);
    }
  }
}

module.exports = { handleIncomingMessage };

