const express = require("express");
const router = express.Router();
const { askVertex } = require("../config/vertex");
const { sendMessageToUser } = require("../services/facebook");
const { handleIncomingMessage } = require("../services/handler");
const config = require("../config/config.json");
/**
 * GET /test - Test all components
 * Returns status of all services
 */
router.get("/", (req, res) => {
  console.log("Testing environment variables...");
  const envCheck = {
    FB_PAGE_ACCESS_TOKEN: config.FB_PAGE_ACCESS_TOKEN ? "✅ Set" : "❌ Missing",
    FB_VERIFY_TOKEN: config.FB_VERIFY_TOKEN ? "✅ Set" : "❌ Missing",
    GCLOUD_PROJECT_ID: config.GCLOUD_PROJECT_ID ? "✅ Set" : "❌ Missing",
    GCLOUD_LOCATION: config.GCLOUD_LOCATION || "us-central1 (default)",
  };

  res.json({
    status: "ok",
    message: "Test endpoint",
    environment: envCheck,
    endpoints: {
      testVertex: "POST /test/vertex - Test Vertex AI",
      testFacebook: "POST /test/facebook - Test Facebook API",
      testHandler: "POST /test/handler - Test complete handler",
      testAll: "POST /test/all - Test everything",
    },
  });
});

/**
 * POST /test/vertex - Test Vertex AI connection
 * Body: { prompt: "your test prompt" }
 */
router.post("/vertex", async (req, res) => {
  try {
    const { prompt } = req.body;
    const testPrompt = prompt || "سلام، چطوری؟";

    console.log("Testing Vertex AI with prompt:", testPrompt);
    const response = await askVertex(testPrompt);

    res.json({
      status: "success",
      message: "Vertex AI test successful",
      prompt: testPrompt,
      response: response,
    });
  } catch (error) {
    console.error("Vertex AI test failed:", error);
    res.status(500).json({
      status: "error",
      message: "Vertex AI test failed",
      error: error.message,
    });
  }
});

/**
 * POST /test/facebook - Test Facebook API
 * Body: { recipientId: "user_id", message: "test message" }
 */
router.post("/facebook", async (req, res) => {
  try {
    const { recipientId, message } = req.body;

    if (!recipientId) {
      return res.status(400).json({
        status: "error",
        message: "recipientId is required",
      });
    }

    const testMessage = message || "این یک پیام تست است 🧪";

    console.log(`Testing Facebook API - Sending to ${recipientId}:`, testMessage);
    await sendMessageToUser(recipientId, testMessage);

    res.json({
      status: "success",
      message: "Facebook API test successful",
      recipientId: recipientId,
      sentMessage: testMessage,
    });
  } catch (error) {
    console.error("Facebook API test failed:", error);
    res.status(500).json({
      status: "error",
      message: "Facebook API test failed",
      error: error.message,
      details: error.response?.data || null,
    });
  }
});

/**
 * POST /test/handler - Test complete handler (Vertex AI + Facebook)
 * Body: { senderId: "user_id", userText: "test message" }
 */
router.post("/handler", async (req, res) => {
  try {
    const { senderId, userText } = req.body;

    if (!senderId) {
      return res.status(400).json({
        status: "error",
        message: "senderId is required",
      });
    }

    const testText = userText || "سلام، تست می‌کنم";

    console.log(`Testing handler - From ${senderId}:`, testText);

    // Call handler (it will send message automatically)
    await handleIncomingMessage(senderId, testText);

    res.json({
      status: "success",
      message: "Handler test successful - Message sent to user",
      senderId: senderId,
      userText: testText,
      note: "Check Facebook Messenger for the response",
    });
  } catch (error) {
    console.error("Handler test failed:", error);
    res.status(500).json({
      status: "error",
      message: "Handler test failed",
      error: error.message,
    });
  }
});

/**
 * POST /test/all - Test everything in sequence
 * Body: { senderId: "user_id", prompt: "optional custom prompt" }
 */
router.post("/all", async (req, res) => {
  const results = {
    vertex: null,
    facebook: null,
    handler: null,
  };

  const { senderId, prompt } = req.body;
  const testPrompt = prompt || "سلام، این یک تست کامل است";
  const testSenderId = senderId || config.GCLOUD_PROJECT_ID;

  if (!testSenderId) {
    return res.status(400).json({
      status: "error",
      message: "senderId is required (or set GCLOUD_PROJECT_ID in env)",
    });
  }

  // Test 1: Vertex AI
  try {
    console.log("Testing Vertex AI...");
    const vertexResponse = await askVertex(testPrompt);
    results.vertex = {
      status: "success",
      response: vertexResponse.substring(0, 100) + "...",
    };
  } catch (error) {
    results.vertex = {
      status: "error",
      error: error.message,
    };
  }

  // Test 2: Facebook API
  try {
    console.log("Testing Facebook API...");
    await sendMessageToUser(testSenderId, "🧪 تست Facebook API");
    results.facebook = {
      status: "success",
      message: "Test message sent",
    };
  } catch (error) {
    results.facebook = {
      status: "error",
      error: error.message,
    };
  }

  // Test 3: Complete Handler
  try {
    console.log("Testing complete handler...");
    await handleIncomingMessage(testSenderId, testPrompt);
    results.handler = {
      status: "success",
      message: "Handler executed successfully",
    };
  } catch (error) {
    results.handler = {
      status: "error",
      error: error.message,
    };
  }

  const allSuccess =
    results.vertex.status === "success" &&
    results.facebook.status === "success" &&
    results.handler.status === "success";

  res.json({
    status: allSuccess ? "success" : "partial",
    message: allSuccess
      ? "All tests passed!"
      : "Some tests failed. Check details below.",
    results: results,
  });
});

module.exports = router;

