# Facebook Messenger Bot with Vertex AI (Gemini)

A Node.js/Express backend that connects Facebook Page Messenger webhooks to Google Vertex AI (Gemini) for automated, intelligent responses.

## 🚀 Quick Start

### 1. Install Dependencies

```bash
npm install
```

### 2. Configure Environment Variables

Copy the example environment file:

```bash
cp .env.example .env
```

Edit `.env` and fill in the required values:

- `PORT` - Server port (default: 3000)
- `FB_VERIFY_TOKEN` - A secret token for webhook verification (create your own)
- `FB_PAGE_ACCESS_TOKEN` - Your Facebook Page Access Token (from Meta Developer Console)
- `GCLOUD_PROJECT_ID` - Your Google Cloud Project ID
- `GCLOUD_LOCATION` - Google Cloud location (default: us-central1)

### 3. Google Cloud Authentication

Set up authentication for Vertex AI:

**Option A: Service Account Key File**
- Create a service account in Google Cloud Console
- Download the JSON key file
- Set the `GOOGLE_APPLICATION_CREDENTIALS` environment variable:
  ```bash
  export GOOGLE_APPLICATION_CREDENTIALS=/path/to/service-account.json
  ```

**Option B: Application Default Credentials**
- If running on Google Cloud (GCE, Cloud Run, etc.), credentials are automatically detected

### 4. Run Locally

```bash
npm run dev
```

The server will start on `http://localhost:3000`

## 🌐 Exposing the Webhook Publicly

Facebook requires a publicly accessible HTTPS endpoint. For local development, use a tunneling service:

### Using ngrok

1. Install ngrok: https://ngrok.com/download
2. Expose your local server:
   ```bash
   ngrok http 3000
   ```
3. Copy the HTTPS URL (e.g., `https://abc123.ngrok.io`)

## 📱 Setting Up Facebook Webhook

1. Go to [Meta for Developers](https://developers.facebook.com/)
2. Select your app and navigate to **Messenger** → **Settings**
3. Under **Webhooks**, click **Add Callback URL**
4. Enter:
   - **Callback URL**: `https://your-domain.com/webhook` (or your ngrok URL)
   - **Verify Token**: The same value as `FB_VERIFY_TOKEN` in your `.env`
5. Subscribe to:
   - `messages`
   - `messaging_postbacks`
6. Click **Verify and Save**

### Getting Page Access Token

1. In Meta Developer Console, go to **Messenger** → **Settings**
2. Under **Access Tokens**, select your Facebook Page
3. Generate a token with `pages_messaging` permission
4. Copy the token to `FB_PAGE_ACCESS_TOKEN` in your `.env`

## 🏗️ Project Structure

```
.
├── src/
│   ├── index.js              # Express app entry point
│   ├── config/
│   │   └── vertex.js         # Vertex AI client configuration
│   ├── services/
│   │   ├── facebook.js       # Facebook Graph API service
│   │   └── handler.js        # Message handler (Vertex AI → Facebook)
│   └── routes/
│       └── webhook.js        # Webhook GET/POST routes
├── .env.example              # Environment variables template
├── .gitignore
├── package.json
└── README.md
```

## 🔧 How It Works

1. **User sends message** → Facebook Messenger
2. **Facebook sends webhook** → POST `/webhook` on your server
3. **Handler processes** → Extracts message text and sender ID
4. **Vertex AI generates response** → Using Gemini model
5. **Response sent back** → Via Facebook Graph API to user

## 📝 API Endpoints

- `GET /` - Health check endpoint
- `GET /webhook` - Webhook verification (Facebook calls this)
- `POST /webhook` - Receives incoming messages from Facebook

## 🚢 Deployment

This backend can be deployed on:

- **VPS** (Ubuntu, etc.) - Use PM2 or systemd
- **Heroku** - Set environment variables in dashboard
- **Google Cloud Run** - Containerized deployment
- **Railway/Render** - Simple PaaS deployment

Make sure to:
- Set all environment variables in your hosting platform
- Ensure `GOOGLE_APPLICATION_CREDENTIALS` is set (or use Application Default Credentials)
- Update Facebook webhook URL to your production domain
- Use HTTPS (required by Facebook)

## 🔒 Security Notes

- Never commit `.env` file
- Keep `FB_VERIFY_TOKEN` secret
- Rotate `FB_PAGE_ACCESS_TOKEN` periodically
- Use environment variables for all sensitive data

## 📚 Dependencies

- **express** - Web framework
- **axios** - HTTP client for Facebook Graph API
- **dotenv** - Environment variable management
- **@google-cloud/vertexai** - Google Vertex AI SDK

## 🐛 Troubleshooting

- **Webhook verification fails**: Check that `FB_VERIFY_TOKEN` matches exactly
- **Messages not received**: Verify webhook subscription includes `messages` event
- **Vertex AI errors**: Check Google Cloud credentials and project permissions
- **Facebook API errors**: Verify `FB_PAGE_ACCESS_TOKEN` is valid and has `pages_messaging` permission

## 📄 License

ISC

